﻿using System;
using System.ServiceModel;

namespace HIPS.Web.Components.ServiceModel
{
    public static class ICommunicationBaseExtensions
    {
        /// <summary>
        /// Safely disposes of a WCF proxy class, addressing the Close/Abort issue.
        /// </summary>
        /// <remarks>
        /// Handles the WCF Using/Close/Abort problem.
        /// Specifically, generated WCF proxies have a Dispose method calling Close.
        /// However, the proxy can be in a faulted state on Dispose leading to Close throwing an exception.
        /// This leaves the WCF connection and proxy in an undisposed faulted state. 
        /// 
        /// See: http://stackoverflow.com/questions/573872/what-is-the-best-workaround-for-the-wcf-client-using-block-issue/
        /// See: http://msdn.microsoft.com/en-us/library/aa355056.aspx
        /// 
        /// See also: http://blogs.msdn.com/b/jjameson/archive/2010/03/18/avoiding-problems-with-the-using-statement-and-wcf-service-proxies.aspx
        /// See also: http://www.codeproject.com/Articles/622989/WCF-and-the-Try-Catch-Abort-Pattern
        /// See also: http://omaralzabir.com/do-not-use-using-in-wcf-client/
        /// See also: http://marcgravell.blogspot.com.au/2008/11/dontdontuse-using.html
        /// See also: http://web.archive.org/web/20100703123454/http://old.iserviceoriented.com/blog/post/Indisposable+-+WCF+Gotcha+1.aspx
        /// See also: http://blog.davidbarrett.net/archive/2007/11.aspx 
        /// See also: http://coding.abel.nu/2012/02/using-and-disposing-of-wcf-clients/
        /// </remarks>
        public static void DisposeSafely(this ICommunicationObject client)
        {
            if (client == null)
            {
                return;
            }

            try
            {
                // If the state is Faulted then Close() will throw
                if (client.State == CommunicationState.Faulted)
                {
                    client.Abort();
                }
                else
                {
                    // This isn't guaranteed to work - the State could still be Faulted here (race condition)
                    client.Close();
                }
            }
            catch (CommunicationException)
            {
                client.Abort();
            }
            catch (TimeoutException)
            {
                client.Abort();
            }
            catch (Exception)
            {
                client.Abort();
                throw; // Rethrow unexpected exception
            }
        }
    }
}
